<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\PluginManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;

class PluginController extends Controller
{
    protected $pluginManager;

    public function __construct(PluginManager $pluginManager)
    {
        $this->pluginManager = $pluginManager;
    }

    /**
     * Display the plugin management dashboard
     */
    public function index()
    {
        $plugins = $this->pluginManager->getAllPlugins();
        $stats = [
            'total' => count($plugins),
            'active' => count(array_filter($plugins, fn($p) => $p['status'] === 'active')),
            'inactive' => count(array_filter($plugins, fn($p) => $p['status'] === 'inactive')),
            'updates' => count(array_filter($plugins, fn($p) => $p['has_update'] ?? false))
        ];

        return view('admin.plugins.index', compact('plugins', 'stats'));
    }

    /**
     * Show plugin installation page
     */
    public function install()
    {
        $plugins = $this->pluginManager->getAllPlugins();
        $stats = [
            'total' => count($plugins),
            'active' => count(array_filter($plugins, fn($p) => $p['status'] === 'active')),
            'inactive' => count(array_filter($plugins, fn($p) => $p['status'] === 'inactive')),
            'updates' => count(array_filter($plugins, fn($p) => $p['has_update'] ?? false))
        ];

        // Get recent installations (you can implement this based on your storage)
        $recentInstallations = $this->getRecentInstallations();

        return view('admin.plugins.install', compact('stats', 'recentInstallations'));
    }

    /**
     * Handle plugin upload and installation
     */
    public function upload(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'plugin' => 'required|file|mimes:zip|max:10240', // Max 10MB
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid file. Please upload a ZIP file under 10MB.',
                    'errors' => $validator->errors()
                ], 422);
            }
            
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $file = $request->file('plugin');
            $result = $this->pluginManager->installPlugin($file->getPathname());

            if ($result['success']) {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => true,
                        'message' => 'Plugin installed successfully!',
                        'plugin' => $result['plugin']
                    ]);
                }
                
                return redirect()->route('admin.plugins.index')
                    ->with('success', 'Plugin installed successfully!');
            } else {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => $result['message'] ?? 'Plugin installation failed.'
                    ], 400);
                }
                
                return redirect()->back()
                    ->with('error', $result['message'] ?? 'Plugin installation failed.');
            }
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred during installation: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->with('error', 'An error occurred during installation: ' . $e->getMessage());
        }
    }

    /**
     * Show detailed plugin information
     */
    public function show($plugin)
    {
        $pluginData = $this->pluginManager->getPlugin($plugin);
        
        if (!$pluginData) {
            abort(404, 'Plugin not found');
        }

        return view('admin.plugins.show', compact('pluginData'));
    }

    /**
     * Activate a plugin
     */
    public function activate($plugin)
    {
        try {
            $result = $this->pluginManager->activatePlugin($plugin);
            
            if ($result) {
                return redirect()->back()->with('success', 'Plugin activated successfully!');
            } else {
                return redirect()->back()->with('error', 'Failed to activate plugin.');
            }
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error activating plugin: ' . $e->getMessage());
        }
    }

    /**
     * Deactivate a plugin
     */
    public function deactivate($plugin)
    {
        try {
            $result = $this->pluginManager->deactivatePlugin($plugin);
            
            if ($result) {
                return redirect()->back()->with('success', 'Plugin deactivated successfully!');
            } else {
                return redirect()->back()->with('error', 'Failed to deactivate plugin.');
            }
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error deactivating plugin: ' . $e->getMessage());
        }
    }

    /**
     * Uninstall a plugin
     */
    public function uninstall(Request $request, $plugin)
    {
        try {
            $result = $this->pluginManager->uninstallPlugin($plugin);
            
            if ($result) {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => true,
                        'message' => 'Plugin uninstalled successfully!'
                    ]);
                }
                return redirect()->route('admin.plugins.index')->with('success', 'Plugin uninstalled successfully!');
            } else {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Failed to uninstall plugin.'
                    ], 400);
                }
                return redirect()->back()->with('error', 'Failed to uninstall plugin.');
            }
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error uninstalling plugin: ' . $e->getMessage()
                ], 500);
            }
            return redirect()->back()->with('error', 'Error uninstalling plugin: ' . $e->getMessage());
        }
    }

    /**
     * Update plugin configuration
     */
    public function updateConfig(Request $request, $plugin)
    {
        try {
            $config = $request->input('config', []);
            $result = $this->pluginManager->updatePluginConfig($plugin, $config);
            
            if ($result) {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => true,
                        'message' => 'Plugin configuration updated successfully!'
                    ]);
                }
                return redirect()->back()->with('success', 'Plugin configuration updated successfully!');
            } else {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Failed to update plugin configuration.'
                    ], 400);
                }
                return redirect()->back()->with('error', 'Failed to update plugin configuration.');
            }
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error updating configuration: ' . $e->getMessage()
                ], 500);
            }
            return redirect()->back()->with('error', 'Error updating configuration: ' . $e->getMessage());
        }
    }

    /**
     * Check for plugin updates
     */
    public function checkUpdates()
    {
        try {
            // For now, return empty updates array
            // You can implement real update checking later
            $updates = [];
            
            return response()->json([
                'success' => true,
                'updates' => $updates,
                'message' => count($updates) > 0 
                    ? count($updates) . ' update(s) available.' 
                    : 'No updates available at this time.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error checking for updates: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a specific plugin
     */
    public function update($plugin)
    {
        try {
            // Implement plugin update logic here
            // For now, return not implemented
            return response()->json([
                'success' => false,
                'message' => 'Plugin updates are not implemented yet.'
            ], 501);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating plugin: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk actions for multiple plugins
     */
    public function bulkAction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'action' => 'required|in:activate,deactivate,uninstall',
            'plugins' => 'required|array|min:1',
            'plugins.*' => 'string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid request data.',
                'errors' => $validator->errors()
            ], 422);
        }

        $action = $request->input('action');
        $plugins = $request->input('plugins');
        $results = [];

        foreach ($plugins as $plugin) {
            try {
                switch ($action) {
                    case 'activate':
                        $results[$plugin] = $this->pluginManager->activatePlugin($plugin);
                        break;
                    case 'deactivate':
                        $results[$plugin] = $this->pluginManager->deactivatePlugin($plugin);
                        break;
                    case 'uninstall':
                        $results[$plugin] = $this->pluginManager->uninstallPlugin($plugin);
                        break;
                }
            } catch (\Exception $e) {
                $results[$plugin] = false;
            }
        }

        $successCount = count(array_filter($results));
        $totalCount = count($plugins);

        return response()->json([
            'success' => $successCount > 0,
            'message' => "Bulk {$action}: {$successCount}/{$totalCount} plugins processed successfully.",
            'results' => $results
        ]);
    }

    /**
     * Export plugin configuration
     */
    public function exportConfig($plugin)
    {
        try {
            $config = $this->pluginManager->exportPluginConfig($plugin);
            
            if ($config) {
                $filename = "{$plugin}_config_" . date('Y-m-d_H-i-s') . '.json';
                
                return response()->json($config)
                    ->header('Content-Disposition', "attachment; filename={$filename}")
                    ->header('Content-Type', 'application/json');
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to export plugin configuration.'
                ], 400);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error exporting configuration: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Import plugin configuration
     */
    public function importConfig(Request $request, $plugin)
    {
        $validator = Validator::make($request->all(), [
            'config_file' => 'required|file|mimes:json|max:1024', // Max 1MB
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid configuration file.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $file = $request->file('config_file');
            $config = json_decode(file_get_contents($file->getPathname()), true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid JSON configuration file.'
                ], 400);
            }

            $result = $this->pluginManager->importPluginConfig($plugin, $config);
            
            if ($result) {
                return response()->json([
                    'success' => true,
                    'message' => 'Plugin configuration imported successfully!'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to import plugin configuration.'
                ], 400);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error importing configuration: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get recent plugin installations
     * 
     * @return array
     */
    private function getRecentInstallations()
    {
        
    }
}